<?php
/**
 * SalesBeast Webinar Page Template
 * Deploy at: webinars.salesbeast.biz/{webinar_id}/index.php
 *
 * Accepts identifiers via URL query params:
 *   ?email=...&phone=...&ip=...
 * Note: server IP is also available via $_SERVER['REMOTE_ADDR'].
 */

declare(strict_types=1);

header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: strict-origin-when-cross-origin');

$path = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
$parts = array_values(array_filter(explode('/', trim($path, '/'))));
// Expect /{webinar_id}/ or /{webinar_id}/index.php
$webinarId = $parts[0] ?? 'default';
$webinarId = preg_replace('/[^a-zA-Z0-9_-]/', '', $webinarId);
if ($webinarId === '') $webinarId = 'default';

$email = isset($_GET['email']) ? trim((string)$_GET['email']) : '';
$phone = isset($_GET['phone']) ? trim((string)$_GET['phone']) : '';
// ip can be passed in URL, but we also capture server-observed IP
$ipFromUrl = isset($_GET['ip']) ? trim((string)$_GET['ip']) : '';
$ipObserved = (string)($_SERVER['REMOTE_ADDR'] ?? '');

// Basic normalization (lightweight; canonical normalization should happen in your app)
$emailNorm = strtolower($email);
$phoneNorm = preg_replace('/[^0-9+]/', '', $phone);
$ipNorm = preg_replace('/[^0-9a-fA-F:\.]/', '', $ipFromUrl);

$ident = [
  'email' => $emailNorm,
  'phone' => $phoneNorm,
  'ip'    => $ipNorm !== '' ? $ipNorm : $ipObserved,
  'ip_observed' => $ipObserved,
];

// Page config (edit per webinar if desired)
$config = [
  // Your SalesBeast backend base URL where the webhook endpoints live
  // Example: https://salesbeast.biz
  'apiBase' => 'https://salesbeast.biz',

  // Protects your webinar webhooks from random traffic
  // Set this to match the token you enforce server-side
  'webhookToken' => 'CHANGE_ME_WEBINAR_TOKEN',

  // How often to send progress pings (seconds)
  'progressIntervalSeconds' => 10,

  // Minimum watch seconds before welcome SMS is eligible (server-side still enforces)
  'minWelcomeWatchSeconds' => 20,

  // Video source (set per webinar)
  // Put an MP4 URL here (BunnyCDN, etc.)
  'videoMp4' => '',

  // Poster image (optional)
  'videoPoster' => '',

  // Page title
  'title' => 'SalesBeast Webinar',
];

?><!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title><?= htmlspecialchars($config['title'] . ' — ' . $webinarId, ENT_QUOTES) ?></title>

  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

  <link rel="stylesheet" href="assets/style.css?v=1" />

  <script>
    window.SB_WEBINAR = {
      webinar_id: <?= json_encode($webinarId) ?>,
      ident: <?= json_encode($ident) ?>,
      apiBase: <?= json_encode($config['apiBase']) ?>,
      webhookToken: <?= json_encode($config['webhookToken']) ?>,
      progressIntervalSeconds: <?= (int)$config['progressIntervalSeconds'] ?>,
      minWelcomeWatchSeconds: <?= (int)$config['minWelcomeWatchSeconds'] ?>,
      videoMp4: <?= json_encode($config['videoMp4']) ?>,
      videoPoster: <?= json_encode($config['videoPoster']) ?>,
      pageUrl: <?= json_encode((isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . ($_SERVER['HTTP_HOST'] ?? '') . ($_SERVER['REQUEST_URI'] ?? '')) ?>
    };
  </script>
</head>

<body>
  <div class="sb-topbar">
    <div class="sb-brand">
      <img class="sb-logo" src="https://salesbeast.biz/app/assets/img/salesbeastlogo-400.png" alt="SalesBeast" />
      <div class="sb-brand-text">
        <div class="sb-brand-title">SalesBeast</div>
        <div class="sb-brand-sub">Live Webinar</div>
      </div>
    </div>

    <div class="sb-topbar-right">
      <div class="sb-pill" id="sb-webinar-id">Webinar: <?= htmlspecialchars($webinarId, ENT_QUOTES) ?></div>
      <button class="sb-btn sb-btn-ghost" id="sb-toggle-chat" type="button" aria-expanded="true">
        <span class="sb-btn-icon">💬</span>
        <span class="sb-btn-text">Chat</span>
      </button>
    </div>
  </div>

  <div class="sb-shell" id="sb-shell">

    <main class="sb-main">
      <section class="sb-video-card">
        <div class="sb-video-frame">
          <video
            id="sb-video"
            class="sb-video"
            controls
            playsinline
            preload="metadata"
            poster="<?= htmlspecialchars($config['videoPoster'], ENT_QUOTES) ?>"
          >
            <?php if (!empty($config['videoMp4'])): ?>
              <source src="<?= htmlspecialchars($config['videoMp4'], ENT_QUOTES) ?>" type="video/mp4" />
            <?php endif; ?>
            Your browser does not support the video tag.
          </video>
          <div class="sb-video-overlay" id="sb-video-overlay">
            <div class="sb-overlay-card">
              <div class="sb-overlay-title">Video not configured</div>
              <div class="sb-overlay-text">
                Set <code>$config['videoMp4']</code> inside <code>index.php</code> for this webinar folder.
              </div>
            </div>
          </div>
        </div>

        <div class="sb-video-meta">
          <div class="sb-meta-left">
            <div class="sb-meta-title">Watch the training</div>
            <div class="sb-meta-sub">Ask questions anytime — Alex will reply here.</div>
          </div>
          <div class="sb-meta-right">
            <div class="sb-meta-badges">
              <span class="sb-badge" id="sb-status-pill">Connecting…</span>
              <span class="sb-badge sb-badge-outline" id="sb-watch-time">0:00</span>
            </div>
          </div>
        </div>
      </section>

      <section class="sb-mobile-chat-trigger" id="sb-mobile-chat-trigger">
        <button class="sb-btn sb-btn-primary" type="button" id="sb-open-chat-mobile">
          Open Chat
        </button>
      </section>

    </main>

    <aside class="sb-chat" id="sb-chat" aria-label="Chat">
      <div class="sb-chat-header">
        <div class="sb-chat-title">
          <div class="sb-chat-name">Alex</div>
          <div class="sb-chat-sub">SalesBeast</div>
        </div>
        <button class="sb-icon-btn" id="sb-collapse-chat" type="button" aria-label="Collapse chat">
          <span aria-hidden="true">⟩</span>
        </button>
      </div>

      <div class="sb-chat-body" id="sb-chat-body" role="log" aria-live="polite"></div>

      <div class="sb-chat-footer">
        <div class="sb-chat-hint">
          You can also text Alex — this chat is synced to your lead record.
        </div>
        <div class="sb-chat-compose">
          <input
            id="sb-chat-input"
            class="sb-chat-input"
            type="text"
            placeholder="Type your question…"
            autocomplete="off"
          />
          <button class="sb-btn sb-btn-primary" id="sb-chat-send" type="button">Send</button>
        </div>
        <div class="sb-chat-mini">
          <div class="sb-mini" id="sb-ident-mini"></div>
        </div>
      </div>
    </aside>

  </div>

  <script src="assets/app.js?v=1"></script>
</body>
</html>
